        PAGE 132
;FILE: FFT-CUT.ASM, Last edition: 02-JAN-1996
;Yet another noise suppressor by SP9VRC - enchanced version of ftspeech.asm
;This one is based on the Fast Fourier Transform
;(c) 1995 Pawel Jalocha, SP9VRC,
;e-mail: jalocha@chopin.ifj.edu.pl, sp9vrc@gw.sp9kby.ampr.org
;
;This software is not to be used for purposes other than amateur radio
;and SWL without the permision of the author.
;Usage for profit or in commercial/military products is explicitly prohibited.
;
;How this filter works:
;Fast Fourier Transform is performed on the input samples.
;Sliding window technique is used to provide data stream continuity.
;This is equivalent to splitting the audio band into several narrow bands.
;Bands with energy below certain threshold are muted completely or partially.
;The Reverse Fourier Transform brings the data back to the time domain.
;Enchancements of the basic idea are:
;1. The threshold is adjusted automatically based on the noise floor which
;   is measured for every frequency band (bin) independently.
;2. Optionally, the filter may request that the particular frequency bin
;   passes the threshold during several consecutive samples -> TrigTimeWidth.
;3. The frequency bin which passed the selection criteria gets "open"
;   so it contributes to the output signal. Optionally it may be left
;   open ahead and afterward for a given number of samples -> TrigHead,TrigTail.
;   Long TrigTail attributes to higher sound fidelity, especially for music.
;Note, that making the threshold dependend on the noise level of a particular
;frequency bin makes the effect of eliminating "dead" carriers. The frequency
;bin where the carrier falls is assumed "noisy" due to having a constant
;strong signal.
;Another thing: the filter works better when the Rx AGC is set in _real_slow_
;mode or when the AGC base is adjusted such that the background noise floor
;is not rising inbetween the voice sounds. This is related to
;the threshold adjustment based on the noise level. When the noise level
;is changing fast, the filter may screw things up...
;That is the reason why I have removed the automatic CODEC's input gain
;adjustment from this code.
;
;For the moment the filter does not contain any arbitrary cuts in frequency
;however, they may be easily implemented by removing (clearing to zero)
;specific frequency components. This way you get a filter passing
;a restricted audio band like 300-3000Hz.
;
;An experimental pulse-type noise suppressor is included in this code.
;
;One comment on the DSPCARD4's noise:
;I still haven't put my card into a metal box, so it makes lot of noise...
;However I have found that connecting two 1nF ceramic capacitors
;from the ground (29,30,31,32,64) to the "-IO" (28) and "+IO" (60) lines
;_greatly_ reduces that noise. I connected the capacitors straight on
;the Euro-connector plug. The numbers given are the pins as on the diagram
;on page 62 of the DSP card 4 User's Manual (Apr-94).
;
;The code contains numerous parameters. The user in encouraged to play with
;them. Infact you can get several very different filters by applying different
;sets of parameters. The initial set was made for best readability of weak
;speech signals with the feature SmoothThreshold enabled.
;An alternative approach:
;SmoothThreshold equ 0
;TrigHead       equ 1
;TrigTail       equ 1
;TrigNeighbours equ 1
;WideClusterScan equ 1 (or more)
;
;Since the 30-OCT-95 I have made one major innovation: some parameters
;can be controlled from the computer via the serial port.
;For example the TrigThreshold can be changed by sending 'T' plus a digit
;followed by the carriage return at 19200 baud like: T4<CR>
;Commands are case sensitive and they are not being echoed.
;For proper commands you will get the answer "OK<CR><LF>",
;for an unknown command X you get "Bad command: X<CR><LF>".
;
;The actuall code starts...

        nolist
	include 'leonid'
	list
        title 'FFT-based noise suppressor by SP9VRC'

EVM56K  equ     1       ;0 => DSPCARD4 by AlefNull
                        ;1 => EVM56K by Motorola (J12 in 16K position)
                        ;the difference is in external X/Y/L RAM allocations

;Base filter parameters
SampleFreq      equ 8000.0      ;8000.0, 9600.0, 16000.0, 32000.0
                                ;audio bandwidth = SampleFreq/2
WindowLen       equ 512 ;sample window length (4..512) - must be a power of 2 !
WindowLenLog    equ @cvi(@log(WindowLen)/@log(2)+0.5)   ;2-base logarythm
                        ;Large window makes better frequency resolution
                        ;but needs more computing power.
FourierLen      equ WindowLen/2
FourierLenLog   equ WindowLenLog-1
FinerShaping    equ 1   ;FFT window with lower sidelobes but the price is
                        ;you have to do twice as many FFT steps.

InlineTables    equ 0   ;make sin/cos tables with the assembler
                        ;if set to 0 the DSP does the job (this saves
                        ;on the .LOD file size and the assembling time)

Stereo          equ 0   ;1=stereo or 0=mono mode.
RightChannel    equ 0   ;1 => for mono mode, use the right input not the left

StereoFreqSepar equ 0   ;artificial stereo effect for CW reception.
                        ;The idea is to project the audio band onto the
                        ;stereo space.
AmplSepar       equ 0   ;activate amplitude-level stereo effect
                        ;The low frequency tones are made stronger
                        ;in the left loudspeaker (headphone)
                        ;while the high tones are made stronger
                        ;in the right loudspeaker.
AmplSeparCenter equ  750.0 ;a tone at this frequency will appear at the center
                           ;of the stereo space (same volume in the left and
                           ;the right speaker).
AmplSeparWidth equ  500.0  ;the band where the volumes for the loudspeakers
                           ;will vary lineary with frequency
                        ;for a narrow CW filter you probably prefer
                        ;Center=750.0, Width=500.0
PhaseSepar      equ 0   ;more sophisticated phase-type stereo effect.
                        ;The _phases_ of various frequencies are shifted such
                        ;that the lower frequencies appear to come from the
                        ;left side of your head and the higher ones appear to
                        ;come from the right side.
PhaseSeparCenter equ  750.0
PhaseSeparWidth  equ  500.0
PhaseSeparBoost equ 4   ;boost the phase-stereo effect by applying larger phase shift
                        ;larger number => heavier boost, 4 => boost=2^4=16

InpGain         equ 0.0 ;The gain in dB's of the CODEC's input
                        ;allowed range: 0..22.5

ExternalEnable  equ 0   ;switch the filter on/off via the PB8 input of the DSP56001
ExternalLearn   equ 0   ;suppress noise floor update via the PB11 input

MedianFollowDown_def equ 7 ;Integration factors for tracing median and noise values
                          ;which represent the estimated noise floor.
MedianFollowUp_def equ 8 ;Especially for CW and FSK signals I suggest that you
                        ;set MedianFollowUp equal to MedianFollowDown+1
                        ;otherwise the noise floor estimatation will treat
                        ;a CW signal as a "dead carrier".
NoiseFollow_def  equ 8  ;for all "Follow" reasonable values are: 3..10
                        ;larger values = slower but more stable noise/median tracing

TrigThreshold_def equ 4 ;0: threshold = noise (in terms of squares)
                        ;1: threshold = 2*noise
                        ;2: threshold = 4*noise
                        ;3: threshold = 8*noise (near squelch effect, but still bleeping)
                        ;4: threshold = 16*noise (full squelch but you can loose weak signals)
                        ;etc...
                        ;higher threshold => better noise removal but you risk
                        ;to make weak signals unreadable or totally unhearable.
WideClusterScan equ 0   ;additional scan(s) for frequency-wide clusters
                        ;possible values: 0,1,2,3
                        ;As this parameter becomes larger it destroys the
                        ;effect of cutting off dead carriers
SmoothThreshold equ 1   ;signal cut-off is smooth that is if below threshold
                        ;the signal will not be cut completely but partially,
                        ;proportionally to its power.
                        ;x = power/(noise*threshold)
                        ;if x>=1 then Pass factor = 1.0
                        ;        else Pass factor = x
                        ;The option in development stage, not fully functional
                        ;comsumes lot of CPU power and destroys partially
                        ;the effect of TrigTimeWidth
                        ;when you activate SmoothThreshold you probably will
                        ;prefer to rise the TrigThreshold for similar noise
                        ;removal effect
                        ;I found experimentally that SmoothThreshold works best
                        ;when TrigNeighbours, TrigTail and TrigHead are set to 0
ParabolicSmooth equ 1   ;parabolic (not linear) threshold smooth.
                        ;x = power/(noise*threshold)
                        ;if x>=1 then Pass factor = 1.0
                        ;        else Pass factor = 2*x - x*x
                        ;When you enable ParabolicSmoth you need to raise
                        ;TrigThreshold by 1 for same noise-removal effect.
                        ;ParabolicSmooth=2 => an experiment:
                        ;if x>=1 then Pass factor = 1.0
                        ;        else Pass factor = (2*x - x*x)^2
CommonThreshold equ 0   ;average the noises for all frequency bins
                        ;and deliver a threshold common for all bins.
                        ;This makes sense if the noise floor is flat.
                        ;In such case you can set the noise/median follow
                        ;factor to very short values and still have a stable
                        ;threshold because it will be averaged over all freq. bins.
TrigNeighbours  equ 0   ;open bins adjacent to the triggered ones
                        ;enable this feature for better sound fidelity
                        ;possible values: 0..any
TrigTimeWidth   equ 0   ;0 = requires just one sample to go above the threshold
                        ;1 = requires two consecutive samples to go above the threshold
                        ;2,3,4,5,6 = etc.
                        ;larger TrigTimeWidth might be good for CW...
TrigHead        equ 0   ;frequency opens that many samples _before_ the trigger
TrigTail        equ 0   ;frequency is left open that many samples _after_ the trigger
                        ;longer head and tail make better sound fidelity.
                        ;Tail is the more important factor.

FourierPipeLen  equ 1+TrigTimeWidth+TrigHead   ;Processing pipe length
TrigMask equ (((1<<TrigTail)-1)<<(FourierPipeLen+1))|((1<<(TrigTimeWidth+TrigHead+1))-1)<<(FourierPipeLen-TrigHead-TrigTimeWidth)

BlankSpikes     equ 0   ;blank spike/pulse-type noise
                        ;I am less and less convinced that
                        ;this option makes any good :-)
SpikeThres      equ 3   ;2 = very strong cut (low threshold)
                        ;3 = effective but distorts the speech
                        ;4 = speech not distorted but pulse noise not
                        ;    completely removed
SpikeAppr       equ 4   ;the number of passes over the data window

ParamControl    equ 1   ;Control filter parameters via the serial interface
                        ;SPY must be disabled for this to work
                        ;For the moment only T<digit> changes the threshold.
CommandLineLen equ 32   ;maximum length of the command line

SPY             equ 0   ;for spying on the noise background
                        ;for the moment works only for WindowLen=512

MonitorHits     equ 0   ;monitor search hits with UP/DOWN/CAT LEDs
                        ;makes little sense unless you connect LEDs
                        ;to the UP/DOWN/CAT outputs for the left tranceiver

;CODEC buffer and process parameters
        if FinerShaping
BatchLen    equ    WindowLen/4    ;processing batch length (must be > 1)
BatchLenLog equ    WindowLenLog-2 ;BatchLen = 2 ^ BatchLenLog
        else
BatchLen    equ    WindowLen/2    ;processing batch length (must be > 1)
BatchLenLog equ    WindowLenLog-1 ;BatchLen = 2 ^ BatchLenLog
        endif
BufLen      equ    WindowLen+BatchLen     ;sample buffer length

;==============================================================================
;control characters

CR equ 13
LF equ 10
NL equ 0

;==============================================================================

UpLED   macro mode      ;UP line (a red LED connected)
        b\mode #1,X:$FFE4
        endm

DownLED macro mode      ;DOWN line (a red LED connected)
        b\mode #2,X:$FFE4
        endm

YellowLED macro mode    ;CAT line (a yellow LED connected)
        b\mode #3,X:$FFE4
        endm

RedLED  macro mode      ; Red LED clr/set/chg
        b\mode #13,X:$FFE4
        endm

;==============================================================================
;FFT macro, copied from Motorola's Dr. BuB.
;alters a,b,x,y, rnm0, rnm1, n2, rnm4, rnm5, rnm6
;Uses 6 locations on System Stack


FFT     macro
         move    #WindowLen/2,n0   ;initialize butterflies per group
         move    #1,n2             ;initialize groups per pass
         move    #WindowLen/4,n6   ;initialize C pointer offset
         move    #-1,m0            ;initialize A and B address modifiers
         move    m0,m1             ;for linear addressing
         move    m0,m4
         move    m0,m5
         move    #0,m6             ;initialize C address modifier for
                                   ;reverse carry (bit-reversed) addressing
;
; Perform all FFT passes with triple nested DO loop
;
         do      #WindowLenLog,_end_pass
         move    #FFTbuff,r0     ;initialize A input pointer
         move    r0,r4           ;initialize A output pointer
         lua     (r0)+n0,r1      ;initialize B input pointer
         move    #FFTcoef,r6     ;initialize C input pointer
         lua     (r1)-,r5        ;initialize B output pointer
         move    n0,n1           ;initialize pointer offsets
         move    n0,n4
         move    n0,n5

         do      n2,_end_grp
         move    x:(r1),x1  y:(r6),y0        ;lookup -sine and 
                                             ; -cosine values
         move    x:(r5),a   y:(r0),b         ;preload data
         move    x:(r6)+n6,x0                ;update C pointer

         do      n0,_end_bfy
         mac     x1,y0,b    y:(r1)+,y1       ;Radix 2 DIT
                                             ;butterfly kernel
         macr    -x0,y1,b   a,x:(r5)+    y:(r0),a
         subl    b,a        x:(r0),b     b,y:(r4)
         mac     -x1,x0,b   x:(r0)+,a  a,y:(r5)
         macr    -y1,y0,b   x:(r1),x1
         subl    b,a        b,x:(r4)+  y:(r0),b
_end_bfy
         move    a,x:(r5)+n5    y:(r1)+n1,y1   ;update A and B pointers
         move    x:(r0)+n0,x1   y:(r4)+n4,y1
_end_grp
         move    n0,b1
         lsr     b   n2,a1     ;divide butterflies per group by two
         lsl     a   b1,n0     ;multiply groups per pass by two
         move    a1,n2
_end_pass
        endm

;==============================================================================
;actuall code in P RAM

        LOMEM P:$0000
        HIMEM P:$1FFF

        scsjmp short            ;short jumps for .if/.else/.endi etc.

        org p:user_code

        jmp <Initialize         ;code to initialize registers, CODEC, arrays,
                                ;is placed at the end so most of the real-time
                                ;code can be placed in internal program RAM.

;------------------------------------------------------------------------------

DoFFT   FFT                     ;a sub-routine for doing the FFT
        rts                     ;takes less program space than having
                                ;the FFT macro at two places

;------------------------------------------------------------------------------

BatchLoop
	waitblk r2,BufLen,BatchLen      ;wait till enough samples for one batch
					;the following code should use r2
					;for addressing the samples
					;r7,m7 must not be used: SSI interrupts
					;r3,m3 must not be used: SCI interrupts and LEONID code
				
Process

          if ExternalEnable
          jclr #8,X:<<$FFE4,SpikeCutDone     ;avoid the cut if PB8 is low
          endif

          if BlankSpikes
            if Stereo
              move r2,r0
              move m2,m0
              jsr <CutSpikes
              move r2,r0
              nop
              move (r0)+
              jsr <CutSpikes
            else
              move r2,r0
              move m2,m0
              if RightChannel
                nop
                move (r0)+
              endif
              jsr <CutSpikes
            endif
          endif
SpikeCutDone
                                ;sliding-window on input
          move n2,n5            ;save n2
          move r2,n4            ;save r2
          move #4*(WindowLen-BatchLen),n2
          move #Window,r5       ;r5 = input window
          move #WindowLen-1,m5
          move #FFTbuff,r4      ;r4 = FFT buffer
          move m5,m4
          move (r2)-n2          ;look backwards in samples
          move n5,n2            ;restore n2
          .loop #WindowLen
            move X:(r2)+,x0 Y:(r5)+,y0  ;x0 = left input, y0 = window
            mpyr x0,y0,a  X:(r2)+n2,x0  ;a = left * window, x0 = right input
            mpyr x0,y0,a  a,X:(r4)
            move a,Y:(r4)+
          .endl
          move n4,r2            ;restore r2

          move n2,Y:<n2save     ;save n2 (FFT modifies it)

          jsr <DoFFT                   ;do the complex FFT

                ;split the complex FT into left/right
          move #FFTbuff,r0
          move #<0,m0
          move #WindowLen/2,n0
          move X:<FourierPipePtr,r1             ;where to place left FT
          move #FourierLen,n1
          move #FourierPipeLen*2*FourierLen-1,m1
          nop
          lua (r1)+n1,r5                        ;where to place right FT
          move m1,m5
          move L:(r0)+n0,y
          move y1,X:(r1)+
          move y0,X:(r5)+
          .loop #WindowLen/2-1
            move L:(r0)+n0,ab
            neg a  ab,L:(r1)+
            move ab,L:(r5)+
          .endl
          move L:(r0)+n0,y
          move (r1)-
          move (r5)-
          .loop #WindowLen/2-1
            move L:(r1),ab
            move L:(r0)+n0,x
            add x1,a
            sub x0,b a,X:(r1)
            move b,Y:(r1)-
            move L:(r5),ab
            add x1,a
            add x0,b a,Y:(r5)
            move b,X:(r5)-
          .endl
          move y1,Y:(r1)
          move y0,Y:(r5)

          ;for mono mode copy left -> right (or right -> left)
          if Stereo==0
           move X:<FourierPipePtr,r1
           move #FourierLen,n1
           move #FourierPipeLen*2*FourierLen-1,m1
           nop
           lua (r1)+n1,r5
           move m1,m5
            if RightChannel
             .loop #FourierLen
               move L:(r5)+,ab
               move ab,L:(r1)+
             .endl
            else
             .loop #FourierLen
               move L:(r1)+,ab
               move ab,L:(r5)+
             .endl
            endif
          endif

          ;now comes the frequency processing...

          ;we try to measure the noise floor on every frequency bin (FreqNoise)
          ;this is not a trivial task... we have to recognize the "good" signal
          ;from the noise.
          move X:<FourierPipePtr,r4
          move #FourierPipeLen*2*FourierLen-1,m4
          move #FourierLen,n4
          move #FreqMedian,r5
          move #FourierLen-1,m5
          move #FreqNoise,r6
          move #FourierLen-1,m6
          move #FreqPower,r1
          move #FourierLen-1,m1
          .loop #FourierLen             ;run over frequency bins
            move X:(r4+n4),x0           ;compute power
            mpy x0,x0,a Y:(r4+n4),x0    ;sum both left and right channels
            mac x0,x0,a X:(r4),x0
            mac x0,x0,a Y:(r4)+,x0
            mac x0,x0,a L:(r5),b        ;load median point
            cmp b,a  a,L:(r1)           ;compare, save the power
            .if <le>            ;if below the median
              tfr b,a           ;decrease the median point
              rep X:<MedianFollowDown ;median-down-follow factor
                asr a           ;if follow-up == follow-down then the median stays at 50% duty
              sub a,b  L:<MinMedian,a     ;check if below minimum ?
              cmp a,b
              tlt a,b
            .else               ;if above the median
              tfr b,a           ;increase the median point
              rep X:<MedianFollowUp ;median-up-follow factor
                asr a
              add a,b
            .endi
            move b,L:(r5)+      ;save the median
            asl b  L:(r1)+,a    ;double the median, load back the power
            asl b
            asl b               ;signal threshold = 8*median
              if ExternalLearn
              jclr #11,X:<<$FFE4,NewNoise     ;avoid the noise update if PB11 is low
              endif
            cmp b,a  L:(r6),b   ;power > signal threshold ?
            .if <le>            ;if power below then
              sub b,a           ;update the noise level
              rep X:<NoiseFollow
                asr a
              add b,a
              move a,L:(r6)
            .endi
NewNoise    move (r6)+
          .endl
;These "follow factors": the larger the factor, the slower the measured
;quantity follows but the less it fluctuates due to the noise

        if ParamControl
          move #AverPower,r5    ;update average spectral power
          move #FourierLen-1,m5
          move L:(r1)+,a        ;a = power
          .loop #FourierLen
            move L:(r5),b       ;b = aver. power
            sub b,a             ;first order low-pass filter
            rep X:<AverPowerFollow
              asr a
            add a,b L:(r1)+,a
            move b,L:(r5)+
          .endl
          move (r1)-
        endif

          ;we spy on the median and noise values
          if SPY
          jsr <SpySync
          jcs <SpyDone
                                ;512 words must follow now
          clr a #FreqMedian,r5
          move #FourierLen-1,m5
          .loop #FourierLen
             move L:(r5)+,a
             rep #8
               asl a
             jsr <SpyA
             nop
          .endl
          clr a #FreqNoise,r5
          .loop #FourierLen
             move L:(r5)+,a
             rep #8
               asl a
             jsr <SpyA
             nop
          .endl
SpyDone
          endif

        if FourierPipeLen>1
          move X:<FourierPipePtr,r4     ;increment the pipe pointer
          move #FourierPipeLen*2*FourierLen-1,m4
          move #2*FourierLen,n4
          nop
          move (r4)+n4
          move r4,X:<FourierPipePtr
        endif

          if CommonThreshold            ;compute the common threshold
            move #FreqNoise,r6          ;if requested
            move #FourierLen-1,m6
            nop
            clr a L:(r6)+,b
            .loop #FourierLen-1
              add b,a L:(r6)+,b
            .endl
            add b,a
            rep #FourierLenLog
              asr a
            move a,L:<CommonThres
          endif

          move #FreqPower,r4    ;r4 = freq. power for this FT sample
          move #FourierLen-1,m4
          move #FreqPower2,r5   ;r5 = smoothed (convoluted) power
          move m4,m5
          move #<2,n5
          move #FreqNoise,r6    ;r6 = noise level of every freq. bin
          move m4,m6
          move #FreqHit,r0      ;hit-flag
          move m4,m0
          move #<2,n0

          move X:<TrigThreshold,a      ;avoid any cuts if threshold is zero
          tst a
          jeq <CutDone

          if MonitorHits
            UpLED clr
          endif
          .loop #FourierLen             ;look for single hits
            if CommonThreshold
             move L:<CommonThres,a
            else
             move L:(r6)+,a     ;load freq. noise
            endif
            rep X:<TrigThreshold
              asl a
            move L:(r4)+,b      ;load freq. power
            cmp a,b Y:(r0),b1   ;compare with threshold
            .if <gt>            ;if above the threshold
              bset #FourierPipeLen,b1           ;then set the hit flag
              if MonitorHits
                UpLED set
              endif
            .endi
            move b1,Y:(r0)+
          .endl

        if WideClusterScan
          move #<1,n4
          move L:(r4)+,b        ;take power and smooth it
          move L:(r4),a         ;load result into power2
          add b,a               ;convolution shape is: 0.5, 1.0, 0.5
          move a,L:(r5)+
          .loop #FourierLen-2
            move L:(r4+n4),a
            add b,a L:(r4)+,b
            addr b,a
            move a,L:(r5)+
          .endl
          move L:(r4)+,a
          add b,a
          move a,L:(r5)+

          if MonitorHits
            DownLED clr
          endif
          .loop #FourierLen             ;look for hits and mark them
            if CommonThreshold
             move L:<CommonThres,a
            else
             move L:(r6)+,a     ;load freq. noise
            endif
            rep X:<TrigThreshold
              asl a
            tfr a,b             ;mult. by 1.5
            asr b
            add b,a  L:(r5)+,b  ;load smoothed freq. power
            cmp a,b             ;compare with threshold
            .if <gt>            ;if above the threshold
              bset #FourierPipeLen,Y:(r0)-      ;then set the hit flag
              bset #FourierPipeLen,Y:(r0)+n0    ;around this cluster
              bset #FourierPipeLen,Y:(r0)-
              if MonitorHits
                DownLED set
              endif
            .endi
            move (r0)+
          .endl

          .loop #FourierLen-1   ;look for 1+1 hits and mark them
            if CommonThreshold
             move L:<CommonThres,a
             rep X:<TrigThreshold
               asl a
            else
             move L:(r6)+,a     ;load freq. noise
             move L:(r6),x      ;load next noise
             add x,a            ;add them
             rep X:<TrigThreshold
               asl a
            endif
            tfr a,b             ;mult. by 1.5
            asr b
            add b,a  L:(r4)+,b  ;load freq. power
            move L:(r4),x       ;load next bin
            add x,b             ;add them
            cmp a,b             ;compare with threshold
            .if <gt>            ;if above the threshold
              bset #FourierPipeLen,Y:(r0)+      ;then set the hit flag
              bset #FourierPipeLen,Y:(r0)-      ;around this cluster
              if MonitorHits
                DownLED set
              endif
            .endi
            move (r0)+
          .endl
          move (r0)+
          move (r4)+
          move (r6)+

        if WideClusterScan>1
          if MonitorHits
            YellowLED clr
          endif
          move L:(r5)+,a          ;more smooth to search for wider
          move L:(r5)-,b          ;clusters
          .loop #FourierLen-1
            add b,a  L:(r5),x
            tfr b,a  a,L:(r5)+n5
            add x,a  L:(r5)-,b
          .endl
          move a,L:(r5)+

          move #<4,n0
          .loop #FourierLen             ;and again look for hits
            if CommonThreshold
             move L:<CommonThres,a
             rep X:<TrigThreshold
               asl a
             rep #2
               asl a
            else
             move L:(r6)+,a     ;load freq. noise
             rep X:<TrigThreshold
               asl a
             rep #2
               asl a
            endif
            move L:(r5)+,b      ;load smoothed freq. power
            cmp a,b             ;compare with threshold
            .if <gt>            ;if above the threshold
              bset #FourierPipeLen,Y:(r0)-      ;then set the hit flag
              bset #FourierPipeLen,Y:(r0)-
              bset #FourierPipeLen,Y:(r0)+n0    ;around this cluster
              bset #FourierPipeLen,Y:(r0)-
              bset #FourierPipeLen,Y:(r0)-
              if MonitorHits
                YellowLED set
              endif
            .endi
            move (r0)+
          .endl

        if WideClusterScan>2
          move L:(r5)+,a          ;further smooth to search for
          move L:(r5)-,b          ;even wider clusters
          .loop #FourierLen-1
            add b,a  L:(r5),x
            tfr b,a  a,L:(r5)+n5
            add x,a  L:(r5)-,b
          .endl
          move a,L:(r5)+

          .loop #FourierLen             ;and again look for hits
            if CommonThreshold
             move L:<CommonThres,a
             rep X:<TrigThreshold
               asl a
             rep #3
               asl a
            else
             move L:(r6)+,a     ;load freq. noise
             rep X:<TrigThreshold
               asl a
             rep #3
               asl a
            endif
            move L:(r5)+,b      ;load smoothed freq. power
            cmp a,b             ;compare with threshold
            .if <gt>            ;if above the threshold
              bset #FourierPipeLen,Y:(r0)-      ;then set the hit flag
              bset #FourierPipeLen,Y:(r0)-      ;then set the hit flag
              bset #FourierPipeLen,Y:(r0)+n0    ;around this cluster
              bset #FourierPipeLen,Y:(r0)-
              bset #FourierPipeLen,Y:(r0)-
              if MonitorHits
                YellowLED set
              endif
            .endi
            move (r0)+
          .endl
        endif
        endif
        endif

          move #FreqOut,r1      ;now mark freq. bins to be turned on
          move m0,m1
          move #>TrigMask,x1
          .loop #FourierLen
            move Y:(r1),a       ;load out-flag
            move Y:(r0),b       ;load hit-flag
            jclr #FourierPipeLen,b1,SaveFlag
              if TrigTimeWidth>0
              jclr #FourierPipeLen-1,b1,SaveFlag
              endif
              if TrigTimeWidth>1
              jclr #FourierPipeLen-2,b1,SaveFlag
              endif
              if TrigTimeWidth>2
              jclr #FourierPipeLen-3,b1,SaveFlag
              endif
              if TrigTimeWidth>3
              jclr #FourierPipeLen-4,b1,SaveFlag
              endif
              if TrigTimeWidth>4
              jclr #FourierPipeLen-5,b1,SaveFlag
              endif
              if TrigTimeWidth>5
              jclr #FourierPipeLen-6,b1,SaveFlag
              endif
                or x1,a
SaveFlag    andi #$FE,ccr       ;clear carry
            ror b               ;shift the hit flag
            andi #$FE,ccr       ;clear carry
            ror a b1,Y:(r0)+    ;shift the out flag, save the hit flag
            move a1,Y:(r1)+     ;save the out flag
          .endl

;          jmp <CutDone          ;*** DEBUG ***

          if ExternalEnable
          jclr #8,X:<<$FFE4,CutDone     ;avoid the cut if PB8 is low
          endif

        if TrigNeighbours
          clr a  #>1,y1         ;a:=0 | y1:=1     a is cluster counter
          move #>2*(TrigNeighbours+1),y0 ;y0:=2*(mark radius +1)
          clr b #<TrigNeighbours,n1      ;b:=0, b will count marked channels

          .loop #TrigNeighbours
            jclr #0,Y:(r1),ExpIniCo     ;jump if not marked
              move y0,a                 ;but if marked then a:=2*(radius+1)
ExpIniCo    move (r1)+
          .endl

          move (r1)-n1
          .loop #FourierLen-TrigNeighbours
            jclr #0,Y:(r1+n1),ExpTst    ;jump if channel not marked
              move y0,a                 ;but if marked then a:=2*(Radius+1)
ExpTst      tst a
            .if <ne>            ;if a!=0 then
              sub y1,a          ;decrease a
              bset #0,Y:(r1)    ;mark channel
              add y1,b          ;count marked channels
            .endi
            move (r1)+
          .endl

          .loop #TrigNeighbours
            tst a
            .if <ne>            ;if a!=0
              sub y1,a          ;decrease a
              bset #0,Y:(r1)    ;mark channel
              add y1,b          ;count marked channels
            .endi
            move (r1)+
          .endl

        endif

          ;now we cut-off frequency bins not marked with FreqOut
          clr a X:<FourierPipePtr,r0
          move #FourierPipeLen*2*FourierLen-1,m0
          move #FourierLen,n0
          .loop #FourierLen
            jset #0,Y:(r1)+,Out_cont
          if SmoothThreshold
            if CommonThreshold
             move L:<CommonThres,a
            else
             move L:(r6),a      ;load noise
            endif
             rep X:<TrigThreshold
               asl a
            tst a L:(r4),b      ;load freq. power
            jeq <Out_cont
            jnr <Norm_done
            jec <Norm_done
            cmpm a,b
            jge <Out_cont
Norm_loop     asl b
              asl a
            jnn <Norm_loop
Norm_done   move a,y0
            move b,a
            rep #24
              div y0,a
          if ParabolicSmooth
            move a0,x0
            mpy x0,x0,b x0,a
            subl b,a
            rnd a
           if ParabolicSmooth>1
            move a,x0
            mpyr x0,x0,a
           endif
            move a,y1
          else
            move a0,y1
          endif
            move L:(r0+n0),x
            mpy x1,y1,a X:(r0),x1
            mpy x0,y1,a a,X:(r0+n0)
            move        a,Y:(r0+n0)
            mpy x1,y1,a Y:(r0),x0
            mpy x0,y1,a a,X:(r0)
            move        a,Y:(r0)
          else
              move a,L:(r0+n0)
              move a,L:(r0)
          endif
Out_cont    move (r0)+
          if SmoothThreshold
            move (r4)+
            move (r6)+
          endif
          .endl

CutDone


        if StereoFreqSepar&&(!Stereo)
         if PhaseSepar
          move X:<FourierPipePtr,r0             ;r0/m0 = I/Q data pointer
          move #FourierPipeLen*2*FourierLen-1,m0 
          move #FourierLen,n0                   ;r0+n0 = right channel
          move #PhaseSeparFactor,r4             ;r4/m4 = phase factors
          move #FourierLen-1,m4
          .loop #FourierLen
            move L:(r0+n0),y
            move L:(r4)+,x
            mpy x1,y1,a
            macr -x0,y0,a
            mpy x1,y0,b
            macr x0,y1,b
            move ab,L:(r0+n0)
            move L:(r0),y
            mpy x1,y1,a
            macr x0,y0,a
            mpy x1,y0,b
            macr -x0,y1,b
            move ab,L:(r0)+
          .endl
         endif
         if AmplSepar
          move X:<FourierPipePtr,r0             ;r0/m0 = I/Q data pointer
          move #FourierPipeLen*2*FourierLen-1,m0 
          move #FourierLen,n0                   ;r0+n0 = right channel
          move #AmplSeparFactor,r4              ;r4/m4 = ampl. factors
          move #FourierLen-1,m4
          move #$7FFFFF,y1                      ;y1 = 1.0 (almost...)
          .loop #FourierLen                     ;loop over freq. bins
            move Y:(r4)+,y0
            move         X:(r0+n0),x0
            mpyr x0,y0,a Y:(r0+n0),x0
            mpyr x0,y0,a a,X:(r0+n0)
            tfr y1,b     a,Y:(r0+n0)
            sub y0,b     X:(r0),x0
            move b,y0
            mpyr x0,y0,a Y:(r0),x0
            mpyr x0,y0,a a,X:(r0)
            move         a,Y:(r0)+
          .endl
         endif
        endif

                ;combine left/right FT into complex one ready for the IFFT
          move #FFTbuff,r0
          move #WindowLen-1,m0
          move X:<FourierPipePtr,r1             ;where to read left FFT
          move #FourierLen,n1
          move #FourierPipeLen*2*FourierLen-1,m1
          nop
          lua (r1)+n1,r5                        ;where to read right FFT
          move m1,m5
          move L:(r1)+,ab
          move a,X:(r0)
          move b,y1
          move L:(r5)+,ab
          move b,y0
          move a,Y:(r0)+
          .loop #WindowLen/2-1
            move L:(r1)+,ab       ;a:b = left FT real/imag.
            neg b  L:(r5)+,x      ;x1:x0 = right FT real/imag.
            add x0,a              ;a = Lr + Ri
            add x1,b              ;b = - Li + Rr
            asr a
            asr b a,X:(r0)
            move b,Y:(r0)+
          .endl
          move (r1)-
          move (r5)-
          move y,L:(r0)+
          .loop #WindowLen/2-1
            move L:(r1)-,ab
            move L:(r5)-,x
            sub x0,a              ;a = Lr - Ri
            add x1,b              ;b = Li + Rr
            asr a
            asr b a,X:(r0)
            move b,Y:(r0)+
          .endl

          jsr <DoFFT            ;do the complex reverse FFT

          move Y:<n2save,n2     ;restore n2

          ;now comes a bit more tricky sliding-window on the output
          move n2,n5            ;save n2
          move #4*(WindowLen-BatchLen),n2
          move #Window,r5       ;r5 = input window
          move #WindowLen-1,m5
          move #FFTbuff,r4      ;r4 = FFT buffer address in bit-reverse order
          move #<0,m4
          move #WindowLen/2,n4
          move (r2)-n2          ;look backwards in samples
          move n5,n2            ;restore n2
          move (r2)+
          .loop #WindowLen-BatchLen
            move L:(r4)+n4,y            ;y1 = left, y0 = right
            move X:(r5)+,x0 Y:(r2)+,a   ;x0 = window. a = left output
            macr y1,x0,a  Y:(r2)-,b     ;a+= left * window, b = right output
            macr y0,x0,b  a,Y:(r2)+     ;b+= right * window, save left
            move b,Y:(r2)+n2
          .endl
          .loop #BatchLen
            move L:(r4)+n4,x            ;x1 = left, x0 = right
            move X:(r5)+,y0             ;y0 = window. a = left output
            mpyr x1,y0,a                ;a = left * window, b = right output
            mpyr x0,y0,b  a,Y:(r2)+
            move b,Y:(r2)+n2
          .endl
          move (r2)-

;          ;we count how many freq. bins is to be output
;          clr a  #>1,x1
;          move #FreqOut,r1
;          move #$FFFF,m1
;          .loop #FourierLen
;            jclr #0,Y:(r1)+,Count_cont
;              add x1,a
;Count_cont  nop
;          .endl
;          move #>'A',x0         ;send out this number
;          add x0,a  #>'Z',x0
;          cmp x0,a
;          tgt x0,a
;          move a,x0
;          putc

      if ParamControl
        jsr <HandleCommands
      endif

        jmp <BatchLoop

;------------------------------------------------------------------------------

        if BlankSpikes

CutSpikes
        move #4*(WindowLen-BatchLen+1),n0
        nop
        move (r0)-n0
        move #<4,n0
        move r0,x1
        clr b X:(r0)+n0,x0              ;measure sum[x^2]
        .loop #WindowLen-1
          mac x0,x0,b X:(r0)+n0,x0
        .endl
        mac x0,x0,b                     ;b=sum[x^2] (total energy)
        .loop #SpikeAppr
          tfr b,a x1,r0                 ;a=energy
          rep #WindowLenLog-SpikeThres  ;scale down to get energy threshold
            asr a                       ;
          move a,L:<SpikeThre           ;save it
          .loop #WindowLen
            move X:(r0)+n0,x0           ;this sample greater than threshold ?
            mac -x0,x0,a
            jge <SpikeNoi_cont          ;jump if not
              mac -x0,x0,b X:(r0)-n0,a  ;if so, subtract this sample from total energy
              move (r0)-n0              ;a = next sample
              move X:(r0)+n0,y0         ;y0 = previous sample
              add y0,a                  ;interpolate
              asr a
              move a,y0                 ;and put back the new value
              mac y0,y0,b y0,X:(r0)+n0  ;add new value to the total energy
SpikeNoi_cont move L:<SpikeThre,a       ;restore energy threshold
          .endl
          nop
        .endl
        rts

        endif

;------------------------------------------------------------------------------

      if ParamControl
HandleCommands
        lookc 0                 ;anything received on the serial port ?
        jcs <HandleCommand_ret  ;jump if nothing
        move #<13,a1                    ;Is this a CR ?
        move #<CommandLineLen-1,m4
        eor x0,a Y:<CommandLinePtr,r4   ;if CR found then the command is
        jeq <ExecCommand                ;complete and we execute it
        move x0,Y:(r4)+                 ;accumulate the incoming characters
        move r4,Y:<CommandLinePtr       ;into the command line
        jmp <HandleCommands

HandleCommand_ret
        rts

ExecCommand
        move a1,Y:(r4)          ;put zero as the last command character
        move #CommandLine,r4    ;start reading the command from the beginning
        move r4,Y:<CommandLinePtr ;reset the command line pointer
        move #$FFFF,m1          ;set m1 for addressing messages with r1
NextCommand
        clr a Y:(r4)+,x0
        eor x0,a
        jeq <Cmd_end
                                ;command dispatch
        move #<'T',a1           ;threshold change ?
        eor x0,a
        jeq <Cmd_Threshold
        move #<'U',a1           ;update factors change ?
        eor x0,a
        jeq <Cmd_UpdateFactors
        move #<'>',a1           ;data sample request ?
        eor x0,a
        jeq <Cmd_DataSample
        move #<'A',a1           ;power average weight change ?
        eor x0,a
        jeq <Cmd_AverPowerFollow
        move #<'?',a1           ;help request ?
        eor x0,a
        jeq <Cmd_Help

Say_BadCmd                      ;send "Bad command: <command><CR><LF>"
        move #Msg_BadCmd,r1
        jsr <SendString
        move Y:-(r4),x0
        putc
        move #Msg_CRLF,r1
        jmp <SendString

Cmd_end
Say_OK  move #Msg_OK,r1
        jmp <SendString

Cmd_Threshold                   ;change the threshold
        jsr <ReadDigit          ;read the following digit
        jcs <Say_BadDigit       ;exit if not a digit
        move b,X:<TrigThreshold ;change the threshold
        jmp <NextCommand        ;read next command in the line

Cmd_UpdateFactors               ;like for CMd_Threshold
        jsr <ReadDigit          ;but somewhere there should be
        jcs <Say_BadDigit       ;the special treatment for zero values.
        move b,X:<NoiseFollow   ;right now the FFT-CUT will crash if you
        jsr <ReadDigit          ;specify zero for any of these update factors
        jcs <Say_BadDigit
        move b,X:<MedianFollowUp
        jsr <ReadDigit
        jcs <Say_BadDigit
        move b,X:<MedianFollowDown
        jmp <NextCommand

Cmd_AverPowerFollow
        jsr <ReadDigit
        jcs <Say_BadDigit
        tst b
        jeq <Say_BadDigit
        move b,X:<AverPowerFollow
        jmp <NextCommand

Cmd_Help
        move #Msg_Help,r1
        jsr <SendString
        jmp <NextCommand

Cmd_DataSample                  ;a data sample was requested
        move Y:(r4)+,x0
        move #<'A',a1
        eor x0,a1
        jeq <Sample_AverPowerLog ;averaged spectral power
        move #<'N',a1
        eor x0,a1
        jeq <Sample_NoiseFloorLog ;estimated noise floor
        move #<'M',a1
        eor x0,a1
        jeq <Sample_SignalMedianLog ;median (auxiliary for the noise floor)
        move #<'F',a1
        eor x0,a1
        jeq <Sample_FourierPowerLog ;momentary spectral power
        jmp <Say_BadCmd

Sample_AverPowerLog             ;send out averaged spectral power in one-byte
        move #>'A',x0           ;logarythmic format
        putc
        move #AverPower,r5
        jmp <SendFourierLen

Sample_SignalMedianLog          ;send out median values in one-byte
        move #>'M',x0           ;logarythmic format
        putc
        move #FreqMedian,r5
        jmp <SendFourierLen

Sample_NoiseFloorLog            ;send out noise floor in one-byte
        move #>'N',x0           ;logarythmic format
        putc
        move #FreqNoise,r5
SendFourierLen
        move #FourierLen-1,m5
        .loop #FourierLen
          move L:(r5)+,a
          jsr <Log2             ;take logarythm
          neg a #<0,x0          ;negate to get positive value
          tmi x0,a              ;but if negative then force to zero
          rep #3                ;scale
            asl a
          move a2,x0            ;and send out on the serial port
          putc
          nop
        .endl
        move #Msg_CRLF,r1
        jsr <SendString
        jmp <NextCommand

Sample_FourierPowerLog          ;send out momentary spectral power in one-byte
        move #>'F',x0           ;logarythmic format
        putc
        move X:<FourierPipePtr,r5
        move #FourierPipeLen*2*FourierLen-1,m5
      if Stereo
        .loop #2*FourierLen
      else
        .loop #FourierLen
      endif
          move X:(r5),x0        ;compute fourier power
          mpy x0,x0,a Y:(r5)+,x0
          mac x0,x0,a
          asl a
          jsr <Log2             ;take logarythm
          neg a #<0,x0          ;negate to get positive value
          tmi x0,a              ;but if negative then force to zero
          rep #3                ;scale
            asl a
          move a2,x0            ;and send out on the serial port
          putc
          nop
        .endl
        move #Msg_CRLF,r1
        jsr <SendString
        jmp <NextCommand

Say_BadDigit
        move #Msg_BadDigit,r1
        jsr <SendString
        move Y:-(r4),x0
        putc
        move #Msg_CRLF,r1
        jmp <SendString

;------------------------------------------------------------------------------
;auxiliary routines for interpreting commands and for sending data

ReadDigit               ;read digit at Y:(r4)
        move Y:(r4)+,b
        move #>'0',x0
        sub x0,b #>9+1,x0
        jcs <ReadDigit_Err
        cmp x0,b
        jcc <ReadDigit_Err
        andi #$FE,ccr
        rts             ;carry=0 => b=digit's value
                        ;x0 is modified, r4 is advanced
ReadDigit_Err
        ori #$01,ccr
        rts             ;carry=1 => not a digit

SendString                      ;send a null-terminated, packed string
        clr a #>$FF,x1          ;extract higher character
        move Y:(r1),a0
        rep #8
          asl a
        eor x1,a1 a1,x0
        jeq <SendString_ret     ;stop if character is NULL
        putc

        clr a #>$FF,x1          ;extract the middle character
        move Y:(r1),a1
        rep #8
          asr a
        and x1,a1
        jeq <SendString_ret     ;stop if character is NULL
        move a1,x0
        putc

        move #>$FF,x1           ;extract lower character
        move Y:(r1)+,a1
        and x1,a1
        jeq <SendString_ret     ;stop if character is NULL
        move a1,x0
        putc

        jmp <SendString

SendString_ret                  ;modifies a,x and b,y,r0/m0 (due to putc)
        rts                     ;r1 is advanced by the number of words
                                ;completely processed

                        ;2-based logarythm function
Log2                    ;a2:1:0 = 56-bit argument (0.0,+256.0)
        tst a #<0,r0    ;r0 ready for finding the integer part
        jmi <Log2_Neg   ;if argument negative...
        jeq <Log2_Inf   ;if zero...
        jnr <Log2_Frac  ;if already normalized
Log2_Norm               ;normalize the argument to the range <0.5,1.0)
        rep #8
          norm r0,a     ;r0 counts the number of shifts performed
        jnn <Log2_Norm
                        ;r0 = approx. integer result
Log2_Frac               ;a = argument scaled to range <0.5,1.0) = X
        clr a a,x0      ;x0 = X
        move r0,a2      ;now we compute the finer fractional approximation
        asr a x0,x1     ;a = approx. integer part
        bchg #23,x0     ;x0 = X-1
        add x0,a        ;a += 2*(X-1) => first, linear correction
        add x0,a
        bchg #22,x1     ;x1 = X-1/2
        mpyr x0,x1,b #0.371,x1 ;b = (X-1)*(X-1/2)
        sub b,a b,x0    ;a += (X-1)*(X-1/2)
        macr -x0,x1,a   ;a += 0.371*(X-1)*(X-1/2)
        rts             ;a = log2(input) <-48.0,+8.0) => second, parabolic corr.
                        ;b,x,r0 are modified
                        ;error RMS is 0.005, maximum error is 0.008
Log2_Neg                ;for zero and negative arguments we give the most
Log2_Inf clr a          ;negative result: a = -48.0
        move #<$E8,a2
        rts             ;execution time is about 80 clock cycles
                        ;but becomes larger for very small arguments

      endif

;------------------------------------------------------------------------------

        if SPY

SpySync move a10,L:<SpySave     ;output: carry=1 => no spy request
        move a2,X:<SpySave+1    ;carry=0 => spy request !
        move x0,Y:<SpySave+1    ;512 words (jsr <SpyA) must follow
        move x1,Y:<SpyCount
        lookc 0
        jcs <Spy_end
        move #>'S',a
        cmp x0,a
        ori #$01,ccr
        jne <Spy_end
        move #>'P',x0
        putc
        move #>512,a
        move a,X:<SpyCount
        andi #$FE,ccr
        jmp <Spy_end

SpyA    move a10,L:<SpySave
        move a2,X:<SpySave+1
        move x0,Y:<SpySave+1
        move x1,Y:<SpyCount
        move X:<SpyCount,a
        tst a
        jne <Spy_copy

Spy_check
        lookc 0
        jcs <Spy_end
        move #>'S',a
        cmp x0,a
        jne <Spy_end
        move #>'P',x0
        putc
        move #>512,a
Spy_copy
        move #>1,x0
        sub x0,a
        move a,X:<SpyCount

        move X:<SpySave,a
	rep	#8
	lsr	a
	move	a1,x0
	putc
        move X:<SpySave,a
	rep	#16
	lsr	a
	move	a1,x0
        putc

Spy_end move L:<SpySave,a10
        move X:<SpySave+1,a2
        move Y:<SpySave+1,x0
        move Y:<SpyCount,x1
        rts

        endif

;------------------------------------------------------------------------------

Initialize
;        andi #%11110011,mr      ;scaling bits = 00

        if InlineTables==0

        move #FFTcoef,r1
        move #WindowLen/2-1,m1
        move #-1.0,a
        move #2.0/WindowLen,b
        move ab,L:<Phase
        .loop #WindowLen/2
           move a,x0
           jsr <IQ
           move ab,L:(r1)+
           move L:<Phase,ab
           add b,a
           move a,X:<Phase
        .endl

        move #Window,r1
        move #WindowLen-1,m1
        clr a #1.0/WindowLen,b
        move ab,L:<Phase
        .loop #WindowLen
           move a,x0
           jsr <IQ
        if FinerShaping
           move b,x0
           mpyr x0,x0,b
        endif
           move b,X:(r1)
           rep #WindowLenLog
             asr b
           move b,Y:(r1)+
           move L:<Phase,ab
           add b,a
           move a,X:<Phase
        .endl

        endif

        clr a #FreqNoise,r4      ;clear the noise array
        move #$FFFF,m4
        .loop #FourierLen
          move a,L:(r4)+
        .endl
        move #FourierPipe,r4    ;clear the Fourier data pipe
        .loop #FourierPipeLen*2*FourierLen
          move a,L:(r4)+
        .endl
        move #FreqHit,r4        ;clear the hit-flags
        .loop #FourierLen
          move a,Y:(r4)+
        .endl
        move #FreqOut,r4        ;and the out-flags
        .loop #FourierLen
          move a,Y:(r4)+
        .endl
        move #Buffer,r4         ;clean the CODEC's buffer
        .loop #4*BufLen         ;if we don't clean it here the filter starts
          move a,L:(r4)+        ;with a big bang, why ?
        .endl                   ;"ctrlcd" (done later) should cleans the buffer !
        move #>$000001,a        ;initialize the median array
        move #FreqMedian,r4
        .loop #FourierLen
          move a,L:(r4)+
        .endl

        if ParamControl
          clr a #AverPower,r4
          .loop #FourierLen
            move a,L:(r4)+
          .endl
        endif

        if StereoFreqSepar&&(!Stereo)
         if PhaseSepar
          clr b #PhaseSeparFactor,r4            ;r4/m4 = separ. coeff. pointer
          move #FourierLen-1,m4                 ;b = freq. of a bin
          move #0.5/@cvf(FourierLen),y1         ;y1 = freq/bin
          move #PhaseSeparCenter/SampleFreq,a   ;a = center freq. for stereo effect
          move #SampleFreq/PhaseSeparWidth/64.0,x1  ;x1 = slope/64
          .loop #FourierLen
            sub a,b b,y0                        ;subtract the center freq.
            move b,x0                           ;mult. by the slope
            mpy x0,x1,b
            rep #6+1                            ;mult. by 64*2
              asl b
            rnd b
            move b,x0                           ;limit to +/-1.0
            mpyr x0,y0,b                        ;mult. by the freq.
            rep #PhaseSeparBoost+1
              asl b
            tfr y0,b b1,Y:(r4)+                 ;save the phase
            add y1,b                            ;increment bin's freq.
          .endl
          .loop #FourierLen             ;convert phase -> I/Q
            move Y:(r4),x0
            jsr <IQ
            move ab,L:(r4)+
          .endl
         endif
         if AmplSepar
          clr b #AmplSeparFactor,r4             ;r4/m4 = separ. coeff. pointer
          move #FourierLen-1,m4                 ;b = freq. of a bin
          move #0.5/@cvf(FourierLen),y1         ;y1 = freq/bin
          move #AmplSeparCenter/SampleFreq,y0      ;y0 = center freq. for stereo effect
          move #SampleFreq/AmplSeparWidth/64.0,x1  ;x1 = slope/64
          .loop #FourierLen
            sub y0,b b,a                        ;subtract the center freq.
            move b,x0                           ;mult. by the slope
            mpy x0,x1,b
            rep #6+1                            ;mult. by 64*2
              asl b
            rnd b
            move b,x0                           ;limit to +/-1.0
            tfr x0,b
            asr b #<0.5,x0                      ;divide by 2, add 0.5
            add x0,b
            tfr a,b b,Y:(r4)+
            add y1,b                            ;increment bin's freq.
          .endl
         endif
        endif

        move #Buffer,r2          ;r2 for us to address the input and output samples
        move #<4-1,n2
        move #BufLen*4-1,m2

        move #Buffer+2,r7        ;r7 for the CODEC's interrupt routine
        move #BufLen*4-1,m7

                        ;initialize input/output control words in the buffer
                        ;zero input/output data
      if EVM56K ;for EVM56002 use the microphone input
        ctrlcd  1,r2,BufLen,MIC,InpGain,InpGain,LINEO|HEADP,0.0,0.0
      else      ;for DSPCARD4 use the LINE input
        ctrlcd  1,r2,BufLen,LINEI,InpGain,InpGain,LINEO|HEADP,0.0,0.0
      endif
        opencd SampleFreq/1000.0,HPF    ;start taking samples at given rate

        jmp <BatchLoop

;------------------------------------------------------------------------------

        if InlineTables==0

PI      equ     3.14159265358979323846

;this routine computes a cosine/sine pair using the sine ROM
;with a second order (linear+quadrature) approximation between table points
IQ                              ;x0 = angle ( -1 = -PI, +1 = +PI)
	ori #%00000100,omr      ;enable the sine ROM table
	move #>$80,x1   ;shift out 8 most significant bits
	mpy x0,x1,a  #>$FF,x0
	move x0,m0
	and x0,a     #>$100,x0
	or x0,a      #<$40,n0
	move a1,r0      ;put the 8 most significant bits into r0 with offset = $100
	move a0,y0      ;save the remaining bits in y0
	jclr #23,y0,SinTable_lev2
	  move (r0)+
SinTable_lev2
	move Y:(r0+n0),x0       ;x0 = coarse cosine
	move Y:(r0),x1          ;x1 = coarse sine
        mpyr x1,y0,a  #PI/256.0,y1
	tfr x0,a  a,x1
	macr -x1,y1,a           ;a = fine cosine
	mpyr x0,y0,b  Y:(r0),x1
        andi #%11111011,omr     ;disable the sine ROM table
        tfr x1,b  b,x1
	macr x1,y1,b  #PI*PI/2.0/65536.0,y1  ;b = fine sine
	mpyr y0,y0,a  a,x0
	move a,y0
	mpyr y0,y1,a
	tfr x0,a  a,y1
	macr -x0,y1,a  b,x1     ;a = super fine cosine
	macr -x1,y1,b           ;b = super fine sine
	rts                     ;x,y are modified
				;r0,m0,n0 are modified
				;maximum error is about 0.7E-6
				;execution time 4+64+4 clock cycles
				;including "jsr <IQ_lev2" and "rts"
        endif

;end of code, data allocation follows

;==============================================================================
;single variables in internal X/Y RAM

        LOMEM X:$0000,Y:$0000,L:$0000
        HIMEM X:$00FF,Y:$00FF,L:$00FF

        org L:user_data

MinMedian dc $000000001000

        if BlankSpikes
SpikeThre ds 1
        endif

        if CommonThreshold
CommonThres ds 1
        endif

        if InlineTables==0
Phase   ds 1
        endif

        if SPY
SpySave dc 0,0
SpyCount dc 0
        endif

LastL = *
        org X:LastL
        org Y:LastL

        org Y:
n2save  ds 1

        org X:
FourierPipePtr dc FourierPipe

        org X:
TrigThreshold dc TrigThreshold_def
NoiseFollow      dc NoiseFollow_def
MedianFollowUp   dc MedianFollowUp_def
MedianFollowDown dc MedianFollowDown_def

        if ParamControl

        org X:
AverPowerFollow dc 5

        org Y:
CommandLinePtr dc CommandLine

        endif

;        org X:
;LastX = *
;        org Y:
;LastY = *
;        if LastX>=LastY
;          org L:LastX
;        else
;          org L:LastY
;        endif

;==============================================================================
;large arrays in external X/Y RAM

        if EVM56K
          LOMEM X:$2000,Y:$0100,L:$2000
          HIMEM X:$3FFF,Y:$3FFF,L:$3FFF
        else
          LOMEM X:$0100,Y:$0100,L:$0100
          HIMEM X:$1FFF,Y:$3FFF,L:$1FFF
        endif

        if EVM56K
          org L:$2000
        else
          org L:$100
        endif

FFTcoef dsm WindowLen/2 ;FFT sine/cosine table

;FreqPower dsm FourierLen
FreqMedian dsm FourierLen
FreqNoise dsm FourierLen
FourierPipe dsm FourierPipeLen*(2*FourierLen)

Buffer  dsm BufLen*4   ;CODEC's input/output buffer
FreqPower               ;we re-use the FFTbuff for FreqPower
FreqPower2 equ FreqPower+FourierLen     ;and for FreqPower2
FFTbuff dsm WindowLen   ;FFT buffer
Window  dsm WindowLen   ;input/output window

      if StereoFreqSepar&&(!Stereo)&&(PhaseSepar)
        org L:
PhaseSeparFactor dsm FourierLen
      endif

      if ParamControl
AverPower dsm FourierLen
      endif

        if EVM56K
          org Y:$100
        else
          org Y:$2000
        endif

        org Y:
FreqHit dsm FourierLen
FreqOut dsm FourierLen

      if StereoFreqSepar&&(!Stereo)&&(AmplSepar)
        org Y:
AmplSeparFactor dsm FourierLen
      endif

      if ParamControl

        org Y:
CommandLine dsm CommandLineLen

Msg_BadCmd dcb 'Bad (sub)command: ',NL
Msg_BadDigit dcb 'Bad digit: ',NL
Msg_OK     dcb 'OK',CR,LF,NL
Msg_CRLF   dcb CR,LF,NL
Msg_Help   dcb 'FFT-CUT, Noise suppressor for speech by SP9VRC',CR,LF,NL

      endif

;==============================================================================
;data allocation done, now initial values for some of the arrays.
;one could let this job be done by the DSP to save the .LOD file size
;and on the assembling time.

        if InlineTables
;Generate sine/cosine table for the FFT

pi      equ     3.141592654
freq = 2.0*pi/@cvf(WindowLen)

        org X:FFTcoef
count   set 0
        dup WindowLen/2
        dc  -@cos(@cvf(count)*freq)
count   set count+1
        endm

        org Y:FFTcoef
count   set 0
        dup WindowLen/2
        dc  -@sin(@cvf(count)*freq)
count   set count+1
        endm

;generate the window shape for sliding-window process

freq = pi/@cvf(WindowLen)

        org Y:Window    ;input window
count   set 0
        dup WindowLen
win = @sin(@cvf(count)*freq)
      if FinerShaping
        dc win*win/@cvf(WindowLen)
      else
        dc win/@cvf(WindowLen)
      endif
count   set count+1
        endm

        org X:Window    ;output window
count   set 0
        dup WindowLen
win = @sin(@cvf(count)*freq)
      if FinerShaping
        dc win*win
      else
        dc win
      endif
count   set count+1
        endm

        endif

;==============================================================================

        end

